
###########################################
# Adds file for entire project
# Examples of project files are:
# dcm_MRID.log
# dcm_MRID_age.log
# dcm_MRID_ageDays.log
# error.log
proc AddProjectFile { filename } {

    if { $::XNATupload(AddProjectFiles) == 0 } {
        return 0
    }
    if {$filename == "" } {
        puts "...>>>>>>ERROR: Got NULL filename. Project file not added."
        return 0
    }


    # check if file exists
    set doit 1
    if { ![ file exists $filename] } {
         puts "...>>>>>>ERROR: Can not find file $filename for project. Not uploading."
        return 0
    }
    if { [ file isdirectory $filename] } {
        puts "...>>>>>>ERROR: $filename appears to be a directory rather than a file. Not uploading."
        return 0
    }

    set fID [exec $::XNAT(client) -host $::XNAT(host) -user_session $::XNAT(sessionID) -m PUT -remote "/REST/projects/$::XNAT(project)/files/$filename" -local "./$filename"]
    # error check
    if { [ErrorCheckOnResponse $fID] == 1 } {
        puts "...>>>>>>ERROR: Got error message instead of resource ID from remote PUT of projectFile=$filename."
        return 0
    } else {
        if { $::XNAT(verbose) } {
            puts "......added/updated $filename for project"
        }
    }

    return 1
}





###########################################
# Adds session files for a session
# Not sure whether these files should be attached to a session, 
# or more appropriately to a reconstruction? or an assessment?
# return 1 if success, 0 if fail
proc AddSessionFile { subjectID expID filename} {

    if { $subjectID == "" } {
         puts "...>>>>>>ERROR: Got NULL subjectID. Not uploading."
        return 0
    }
    if { $expID == "" } {
         puts "...>>>>>>ERROR: Got NULL experimentID. Not uploading."
        return 0
    }
    if { $filename == ""} {
         puts "...>>>>>>ERROR: Got NULL filename. Not uploading."
        return 0
    }
    if { ![ file exists $filename] } {
        return 0
    }
    if { [ file isdirectory $filename] } {
        return 0
    }

    
    set fID [exec $::XNAT(client) -host $::XNAT(host) -user_session $::XNAT(sessionID) -m PUT -remote "/REST/projects/$::XNAT(project)/subject/$subjectID/experiment/$expID/resources/sessionFiles/files/$filename?format=ASCIItext" - local "./$filename" ]
    # error check
    if { [ErrorCheckOnResponse $fID] == 1 } {
        puts "...>>>>>>ERROR: Got error message instead of resource ID from remote PUT of sessionFile=$filename."
        return 0
    } 

    if { $::XNAT(verbose) } {
        puts "......added/updated $filename to sessionFile resource."
    }
    return 1
}





###########################################
# Adds session files for a session 
# Not sure whether these files should be attached to a session, 
# or more appropriately to a reconstruction? or an assessment?
# Session files are listed in global ::XNAT(sessionFileNames)
# return 1 if success, 0 if abort, -1 if fail
proc AddSessionFiles { subjectID experimentLabel } {

    if { $::XNATupload(AddSessionFiles) == 0 } {
        return
    }
    if { $subjectID == "" } {
        puts "...>>>>>>ERROR: Got a NULL subjectID. No session files are added."
        return -1
    }
    if {$experimentLabel == "" } {
        puts "...>>>>>>ERROR: Got a NULL MRsession label. No session files are added."
        return -1
    }
    set expID [ QueryForExperimentID $subjectID $experimentLabel ]
    if { $expID <= 0} {
        puts "...>>>>>>ERROR: Could not find experimentID for label $experimentLabel. No session files are added."
        return -1
    }
    
    #--- create a resource for the group of sessionFiles:
    set xml [ exec $::XNAT(client) -host $::XNAT(host) -user_session $::XNAT(sessionID) -m PUT -remote "/REST/projects/$::XNAT(project)/subject/$subjectID/experiment/$expID/resources/sessionFiles?xnat:resourceData/label=sessionFiles"] 
    if { $xml == "" } {
        puts "...Created resource to contain sessionFiles"
        set len [ llength $::XNAT(sessionFileNames)]
        for { set i 0 } { $i < $len } { incr i } {
            set fname [ lindex $::XNAT(sessionFileNames) $i ]
            if { [file exists $fname] } {
                AddSessionFile $subjectID $expID $fname 
            }
        }
    } else {
        puts "...>>>>>>ERROR: Could not create a resource to contain the group of session files. No session files added or updated."
        return -1
    }
    return 1
}






###########################################
# creates a Scan on XNAT to hold scan data
# return 1 if success, 0 if error.
proc AddScan { subjectID expID sID type } {

    if { $subjectID == "" } {
        puts "...>>>>>>ERROR: Got a NULL subjectID. No scan is created to contain scan data."
        return 0
    }
    if {$expID == "" } {
        puts "...>>>>>>ERROR: Got a NULL experimentID. No scan is created to contain scan data."
        return 0
    }
    if {$sID == "" } {
        puts "...>>>>>>ERROR: Got a NULL scanID. No scan is created to contain scan data."
        return 0
    }

    if { $type == "" } {
        set type "unknown"
    }
    
    set xml [exec $::XNAT(client) -host $::XNAT(host) -user_session $::XNAT(sessionID) -m PUT -remote "/REST/projects/$::XNAT(project)/subjects/$subjectID/experiments/$expID/scans/$sID?xnat:mrScanData/type=$type"]
    if {$xml == "" } {
            return 1
    }

    return 0
}





###########################################
# interface to XNATRestClient to add all files for a given scan
# return 1 if success, 0 if aborted -1 if error
proc AddScanData { subjectID expID sname type format fname  } {

    #fname is first name of file set. Get all files with the same prefix.
    #expect names that look something like: xxxxxx-xxxxxxx-xxxxxxxxx.dcm
    set completeUpload 1
    set index [ string first ".dcm" $fname ]
    if { $index >= 0 } {
        set base [ string range $fname 0 [expr $index -1 ]]
        set index [ string last "-" $base]
        if { $index >= 0 } {
            set base [ string range $base 0 $index]
        }
    
        # can we find a set of (at least one) files?
        if { [catch {glob $base*} ] == 0 } {
            set fileList [ glob $base* ]
            set len [ llength $fileList ]
            for { set i 0 } { $i < $len } { incr i } {
                set dicomfile [ lindex $fileList $i ]
                puts "...Adding file to scan: $dicomfile"
                set xml [exec $::XNAT(client) -host $::XNAT(host) -user_session $::XNAT(sessionID) -m PUT -remote "/REST/projects/$::XNAT(project)/subjects/$subjectID/experiments/$expID/scans/$sname/files/$dicomfile?format=$format\&content=$type" -local "./$dicomfile" ]
                if { $xml != "" } {
                    set completeUpload 0
                }
            }
            if { $completeUpload == 0 } {
                puts "...>>>>>>ERROR: AddScanData got unexpected response from remote PUT of scan data: The scan may not have been properly created. Perform QA check for:  subject=$subjectID MRSession=$expID scan=$sname on remote host."            
                return 0
            }
            return 1
        }
    }

    puts "...>>>>>>ERROR: AddScanData could not extract local set of files for scan $sname."
    return 0
}






###########################################
# interface to XNATRestClient to add all scan data for a subject
# return 1 if success, 0 if aborted -1 if error
proc AddAllScanData {subjectID experimentLabel } {
    
    if { $::XNATupload(AddAllScanData) == 0 } {
        return
    }

    if { $subjectID == "" } {
        puts "...>>>>>>ERROR: Got a NULL subjectID. No scans are uploaded."
        return
    }
    if {$experimentLabel == "" } {
        puts "...>>>>>>ERROR: Got a NULL MRsession label. No scans are uploaded."
        return
    }
    set expID [ QueryForExperimentID $subjectID $experimentLabel ]
    if { $expID <= 0} {
        puts "...>>>>>>ERROR: could not find experimentID for label $experimentLabel. No scans are added."
        return
    }

    #--- get list of scans out of toc.txt
    # cd into directory and parse toc.txt (if it exists) for metadata
    if { [GetScanList]  == 0 } {
        puts "...>>>>>>WARNING: Did not retreive any scans from directory $experimentLabel. No scans uploaded."
        return 0
    }

    # make sure lists  will work.
    set nlen [llength $::XNATupload(scanNames)]
    set tlen [llength $::XNATupload(scanTypes)]
    if { $nlen == 0 || $tlen == 0 || $nlen != $tlen } {
        puts "...>>>>>>ERROR: Did not retrieve good results from parsing of scan names and types."
        return 0
    }
    
    # create a unique ID for this scan.
    # we don't know how many scans are alread present.
    set uniqueID 0
    for { set i 0 } { $i < $nlen } { incr i } {
        set quit 0
        while { $quit == 0 } {
            incr uniqueID
            set quit [ QueryForUniqueScanID $subjectID $expID "Scan$uniqueID" ]
            if { $quit == -1 } {
                puts "...>>>>>>ERROR: Unable to verify whether proposed scanID is unique. No scans uploaded for subject $subjectID and $experiment $expID."
                break
            }
        }
        # upload scans if we got a unique ID
        if { $quit > 0 } {
            set sname "Scan$uniqueID"
            set fname [lindex $::XNATupload(scanNames) $i ]
            set type [ lindex $::XNATupload(scanTypes) $i ]
            #--- first, create the scan to contain all files.
            puts "\n"
            puts "---------------------------------------------------------------------------"
            puts "......Creating scan to contain files for this MRSession"
            puts "---------------------------------------------------------------------------"
            if {[AddScan $subjectID $expID $sname $type] > 0 } {
                if { [ string first ".dcm" $fname ] >= 0 } {
                    set format "DICOM"
                } else {
                    set format "unspecified"
                }
                puts "\n"
                puts "---------------------------------------------------------------------------"
                puts "......Adding all files to scan"
                puts "---------------------------------------------------------------------------"
                AddScanData $subjectID $expID $sname $type $format $fname
            } else {
                puts "...>>>>>>ERROR: Unable to create scan $sname on XNAT to contain image data. No scan uploaded."                
            }

        }
    }
}





###########################################
# interface to XNATRestClient to add a new subject 
# return 1 if success, 0 if aborted -1 if error
proc AddSubject {subjectID } {

    if { $subjectID == "" } {
        puts "...>>>>>>ERROR: Got a NULL subjectID. No subject is created."
        return -1
    }

    set q [ QueryForSubject $subjectID ]
    if {  $q == 1 } {
        puts "...>>>>>>WARNING: Subject $subjectID exists; not re-adding.."
        return 0
    } elseif {  $q == -1 } {
        puts "...>>>>>>ERROR: Not adding subject $subjectID."
        return -1
    }
    
    # no subject found in project.
    if { $::XNAT(verbose) } {
        puts "...Creating subject on XNAT"
    }

    set sID [exec $::XNAT(client) -host $::XNAT(host) -user_session $::XNAT(sessionID) -m PUT -remote "/REST/projects/$::XNAT(project)/subjects/$subjectID" ]
    
    if { [ErrorCheckOnResponse $sID] > 0 } {
        puts "...>>>>>>ERROR: Got error message instead of subjectID from remote PUT of subject: Subject may not have been properly created. Perform QA check for subject=$subjectID on remote host."
        return -1       
    }

    if { $::XNAT(verbose) } {
        puts "......Created subjectID $subjectID within project $::XNAT(project)"
    }
    return 1
}





###########################################
# interface to XNATRestClient to describe experiment
# return experimentID if success, 0 if aborted -1 if error
proc AddExperiment {subjectID experimentLabel } {

    if { $subjectID == "" } {
        puts "...>>>>>>ERROR: Got a NULL subjectID. No experiment is created."
        return -1
    }
    
    if { $experimentLabel == "" } {
        puts "...>>>>>>ERROR: Got a NULL experiment label. No experiment is created."
        return -1
    }

    set q [ QueryForSubject $subjectID ]
    if {  $q <= 0 } {
        puts "...>>>>>>ERROR: No subject $subjectID found. Not adding experiment for this subject."
        return -1
    } 

    set q [QueryForExperiment $subjectID $experimentLabel ]
    if { $q == 1 } {
        puts "...>>>>>>WARNING: Experiment already exists for subject $subjectID. Not re-creating."
        return 0
    } elseif { $q == -1 } {
        puts "...>>>>>>ERROR: Not adding experiment $experimentLabel"
        return -1
    }
    
    if { $::XNAT(verbose) } {        
        puts "...$experimentLabel not found; creating new experiment with this label...."
    }

    # create an experiment (specifically an MRSession) with the sessionName as a label.
    set expID [exec $::XNAT(client) -host $::XNAT(host) -user_session $::XNAT(sessionID) -m PUT -remote "/REST/projects/$::XNAT(project)/subjects/$subjectID/experiments/MRSession?xnat:mrSessionData/label=$experimentLabel" ]
    
    if { [ErrorCheckOnResponse $expID] > 0 } {
        puts "...>>>>>>ERROR: Got error message instead of experimentID from remote PUT of experiment: Experiment may not have been properly created. Please perform QA check for subject=$subjectID experimentLabel=$experimentLabel on remote host."
        return -1
    }

    if { $::XNAT(verbose) } {        
        puts "...Created new experiment with ID=$expID and label=$experimentLabel"
    }

    # return the new experiment ID
    return $expID
}






###########################################
# interface to XNATRestClient to describe project
# return 1 if success, 0 if aborted -1 if error
proc AddProjectMetadata {mrid } {
}






###########################################
# interface to XNATRestClient to describe subject
# return 1 if success, 0 if no subject was found -1 if error
proc AddSubjectMetadata {subjectID experimentLabel} {
    
    if { $subjectID == "" } {
        puts "...>>>>>>ERROR: Got a NULL subjectID. No subject metadata is added."
        return 0
    }
    if { $experimentLabel == "" } {
        puts "...>>>>>>ERROR: Got a NULL experiment label. No subject metadata is added."
        return -1
    }

    # check to see if directory is present
    if { ![ file exists $experimentLabel] } {
        puts "Can not find directory $experimentLabel -- not creating experiment for this."
        return -1
    }
    if { ![ file isdirectory $::XNAT(localDataDirectory)] } {
        puts "$experimentLabel appears not to be a directory -- not creating experiment for this"
        return -1
    }

    set q [ QueryForSubject $subjectID ] 
    if {$q  <= 0 } {
        puts "...>>>>>>ERROR: Not adding metadata for subject $subjectID."
        return 0
    }

    
    # cd into directory and parse toc.txt (if it exists) for metadata
    if { [ParseTOC $experimentLabel]  == 0 } {
        return -1
    }

    #
    #--- build out subject demographic information uri
    #--- try two ways;


    if { 0 } {
        #--- year of birth
        if { $::XNATupload(yob) != "" } {
            set subjectData "$::XNATupload(yobPrefix)$::XNATupload(yob)"
            set retval [exec $::XNAT(client)  -host $::XNAT(host) -user_session $::XNAT(sessionID) -m PUT -remote "/REST/projects/$::XNAT(project)/subjects/$subjectID$subjectData"]
            if { [ErrorCheckOnResponse $retval] > 0 } {
                puts "...>>>>>>ERROR: Did not get expected response from server. May not have added year of birth information to subject"
                return -1      
            }
        }
        #--- gender
        if { $::XNATupload(gender) == "male" || $::XNATupload(gender) == "female"} {
            puts "...adding demographic data to subject $subjectID"
            #--- add demographics data to the subject if something to add
            set subjectData "$::XNATupload(genderPrefix)$::XNATupload(gender)"
            set retval [exec $::XNAT(client)  -host $::XNAT(host) -user_session $::XNAT(sessionID) -m PUT -remote "/REST/projects/$::XNAT(project)/subjects/$subjectID$subjectData"]
            if { [ErrorCheckOnResponse $retval] > 0 } {
                puts "...>>>>>>ERROR: Did not get expected response from server. May not have added gender information to subject"
                return -1
            }
        }
        #--- date of birth
        if { $::XNATupload(dob) != "" } {
            set subjectData "$::XNATupload(dobPrefix)$::XNATupload(dob)"
            set retval [exec $::XNAT(client)  -host $::XNAT(host) -user_session $::XNAT(sessionID) -m PUT -remote "/REST/projects/$::XNAT(project)/subjects/$subjectID$subjectData"]
            if { [ErrorCheckOnResponse $retval] > 0 } {
                puts "...>>>>>>ERROR: Did not get expected response from server. May not have added date of birth information to subject"
                return -1
            }
        }
    }


    if { 1 } {
        set fp [ open "./subjectDemographics.xml" "w+"]
        
        puts $fp "<xnat:Subject ID=\"$subjectID\" project=\"$::XNAT(project)\" group=\"\" label=\"\" src=\"\" xmlns:xnat=\"http://nrg.wustl.edu/xnat\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\">"
        puts $fp "<xnat:demographics xsi:type=\"xnat:demographicData\">"
        puts $fp "   <xnat:yob>$::XNATupload(yob)</xnat:yob>"
        puts $fp "   <xnat:dob>$::XNATupload(dob)</xnat:dob>"
        puts $fp "   <xnat:gender>$::XNATupload(gender)</xnat:gender>"
        puts $fp "   <xnat:handedness></xnat:handedness>"
        puts $fp "   <xnat:education></xnat:education>"
        puts $fp "   <xnat:race></xnat:race>"
        puts $fp "   <xnat:ethnicity></xnat:ethnicity>"
        puts $fp "   <xnat:weight></xnat:weight>"
        puts $fp "   <xnat:height></xnat:height>"
        puts $fp "</xnat:demographics>"
        puts $fp "</xnat:Subject>"
        close $fp
        set retval [exec $::XNAT(client)  -host $::XNAT(host) -user_session $::XNAT(sessionID) -m PUT -remote "/REST/projects/$::XNAT(project)/subjects/$subjectID" -local ./subjectDemographics.xml]

        if { [ErrorCheckOnResponse $retval] > 0 } {
            puts "...>>>>>>ERROR: Did not get expected response from server. May not have added subject demographics for subject=$subjectID."
            return -1
        }
        #--- clean up
        file delete -force -- "./subjectDemographics.xml"
    }

    #--- success.
    return 1
}






###########################################
# interface to XNATRestClient to describe experiment
# return 1 if success, 0 if aborted -1 if error
proc AddExperimentMetadata {expID} {


    if { $expID == ""} {
        puts "...>>>>>>ERROR: Got a NULL experimentID. No experiment metadata is added."
        return -1
    }
    
    # check to see if directory is present
    if { ![ file exists $expID] } {
        #TESTDATA_DEBUG_COMMENT
        #puts "...>>>>>>ERROR: Can not find directory $experimentLabel -- not creating experiment for this."
        return -1
    }

    if { ![ file isdirectory $::XNAT(localDataDirectory)] } {
        #TESTDATA_DEBUG_COMMENT
        #puts "...>>>>>>ERROR: $experimentLabel appears not to be a directory -- not creating experiment for this"
        return -1
    }

    set q [QueryForExperiment $subjectID $experimentLabel ] 
    if {$q  <= 0 } {
        #TESTDATA_DEBUG_COMMENT
        #puts "...>>>>>>ERROR: ...Not adding metadata for experiment $experimentLabel"
        return 0
    }

    # cd into directory and parse toc.txt (if it exists) for metadata
    if { [ParseTOC $expID]  == 0 } {
        return 0
    }

    #
    #--- build out experiment details we may want to search on later...
    #--- (ideally append all demogs together, but can't seem to get it to work.
    #--- so for now do separate calls for each -- slower but works.)
    #

    #--- study date
    set expData $::XNATupload(studydatePrefix)
    if { $::XNATupload(studydate) != "" } {
        set expData "$expData$::XNATupload(studydate)"
        set retval [exec $::XNAT(client)  -host $::XNAT(host) -user_session $::XNAT(sessionID) -m PUT -remote "/REST/projects/$::XNAT(project)/subjects/$subjectID/experiments/$expID$expData"]
        if { [ErrorCheckOnResponse $retval] > 0 } {
            puts "...>>>>>>ERROR: Did not get expected response from server. May not have added study date information to Experiment"
            return -1
        }
    }

    #---modality
    if { $::XNATupload(modalityPrefix) != "" } {
        set modality "MR"
        set expData "$::XNATupload(modalityPrefix)$modality"
        set retval [exec $::XNAT(client)  -host $::XNAT(host) -user_session $::XNAT(sessionID) -m PUT -remote "/REST/projects/$::XNAT(project)/subjects/$subjectID/experiments/$expID$expData"]
        if { [ErrorCheckOnResponse $retval] > 0 } {
            puts "...>>>>>>ERROR: Did not get expected response from server. May not have added modality information to Experiment"
            return -1
        }
    }

    #---age in years
    if { $::XNATupload(age) != "" } {
        set expData "$::XNATupload(agePrefix)$::XNATupload(age)"
        set retval [exec $::XNAT(client)  -host $::XNAT(host) -user_session $::XNAT(sessionID) -m PUT -remote "/REST/projects/$::XNAT(project)/subjects/$subjectID/experiments/$expID$expData"]
        if { [ErrorCheckOnResponse $retval] > 0 } {
            puts "...>>>>>>ERROR: Did not get expected response from server. May not have added modality information to Experiment"
            return -1
        }
    }

    #---scanner
    if { $::XNATupload(scannermfg) != "" } {
        #--- replace spaces with underbars
        set idlist [ lsearch -all $::XNATupload(scannermfg) " " ]
        set idlistlen [ llength $idlist ]
        for { set index 0 } { $index < $idlistlen } { incr index } {
            set ::XNATupload(scannermfg) [lreplace $::XNATupload(scannermfg) $index $index "_"]
        }
    }
    #--- only add the model if we have a manufacturer
    if { ( $::XNATupload(scannermfg) != "") && ( $::XNATupload(scannermod) != "") } {
        #--- replace spaces with underbars
        set idlist [ lsearch -all $::XNATupload(scannermod) " " ]
        set idlistlen [ llength $idlist ]
        for { set index 0 } { $index < $idlistlen } { incr index } {
            set ::XNATupload(scannermod) [ lreplace $::XNATupload(scannermod) $index $index "_"]
        }
        #--- qualify scanner name with model name if we have it.
        set ::XNATupload(scannermfg) "$::XNATupload(scannermfg)_$::XNATupload(scannermod)"
    }
    if {$::XNATupload(scannermfg) != "" } {
        set expData"$::XNATupload(scannerPrefix)$::XNATupload(scannermfg)"
        set retval [exec $::XNAT(client)  -host $::XNAT(host) -user_session $::XNAT(sessionID) -m PUT -remote "/REST/projects/$::XNAT(project)/subjects/$subjectID/experiments/$expID$expData"]
        if { [ErrorCheckOnResponse $retval] > 0 } {
            puts "...>>>>>>ERROR: Did not get expected response from server. May not have added scanner information to Experiment"
            return -1
        }
    }

    #--- success.
    return 1
}




###########################################
# interface to XNATRestClient to describe reconstructions
# return 1 if success, 0 if aborted -1 if error
proc AddReconstructedImageData {string MRID} {
}


###########################################
# interface to XNATRestClient to describe assessors
# return 1 if success, 0 if aborted -1 if error
proc AddAssessorScanData {string MRID} {
}





###########################################
# This is a custom proc for Rudolph & Ellen's dataset.
# Its basic approach can be used to guide other bulk upload
# usecases.
#
# Uploads entire local retrospective data;  assumes calling script 
# has cd'ed into the project's root data directory.
# This method *should* pass over all data already uploaded and just
# upload new stuff.
#
proc BulkUpload { } {

    ##
    ## map project structure onto xnat's data model.
    ##    
    CreateProjectToDataModelMapping


    ##
    ## Create remote session
    ## Get one sessionID that is good for many transactions.
    ##
    puts "\n"
    puts "---------------------------------------------------------------------------"
    puts "Creating remote session with XNAT web services."
    puts "---------------------------------------------------------------------------"
    CreateXNATSession $::XNAT(username) $::XNAT(password)
    if {[ ErrorCheckOnResponse $::XNAT(sessionID)] > 0}  {
        puts "...>>>>>>ERROR: Got error from XNAT while trying to create a session ID"
        exit 1
    }




    ##
    ## Check to see if project is valid
    ##
    puts "\n"
    puts "---------------------------------------------------------------------------"
    puts "Checking xnat to see if $::XNAT(project) exists..."
    puts "---------------------------------------------------------------------------"
    if { [QueryForProject $::XNAT(project)] < 1 } {
        puts "...>>>>>>ERROR: Project $::XNAT(project) was not found on $::XNAT(host)"
        exit 1
    } else {
        puts "...confirmed $::XNAT(project) exists."
    }




    ##
    ## For a valid project, upload four projectFiles:
    ## 
    puts "\n"
    puts "---------------------------------------------------------------------------"
    puts "Adding project files."
    puts "---------------------------------------------------------------------------"
    #--- if project resource was created, add project files.
    set len [ llength $::XNAT(projectFileNames)]
    for { set i 0} {$i < $len } {incr i } {
        set pfile [ lindex $::XNAT(projectFileNames) $i ]
        puts "...adding/updating $pfile"
        AddProjectFile $pfile
    }




    ##
    ## Create containers for subjects if they don't already exist
    ## create subjects in main loop: set ::XNATupload(CreateAllSubjects) 1
    ## skip creation in main loop: set ::XNATupload(CreateAllSubjects) 0
    ##
    if { $::XNATupload(CreateAllSubjects) } {
        puts "\n"
        puts "---------------------------------------------------------------------------"
        puts "Creating new subjects in project $::XNAT(project) on XNAT"
        puts "---------------------------------------------------------------------------"
        set numMRIDs [llength $::XNAT(MRIDs)]
        for { set i 0 } { $i < $numMRIDs } { incr i } {
            #--- add the subject
            set mrid [ lindex $::XNAT(MRIDs) $i ]
            # if subject was added, update metadata
            if { [AddSubject $mrid] > 0 } {
                #--- extract & add subject metadata from toc file in each session dir for subject
                #--- hmmm, i guess last session dir for subject gets final say...
                set numSessions [llength $::XNATupload($mrid,MRSessionDirnameList)]
                for { set j 0} { $j < $numSessions } { incr j } {
                    set sessionLabel [ lindex $::XNATupload($mrid,MRSessionDirnameList) $j ]
                    if {[ AddSubjectMetadata $mrid $sessionLabel] <= 0 } {
                        puts "......No demographic data was added for $mrid for project $sessionLabel."
                    }
                }
            }
        }
    } else {
        puts "...(skipping creation of all subjects)"
    }



    
    ##
    ## For each subject:
    ## 1. create one or more experiments (MRSession) with the directory name as label,
    ## 2. and add experiment metadata
    ## 3. upload SessionFiles
    ## 4. upload Session scan data    
    ##
    if { $::XNATupload(CreateAllMRSessions) } {
        puts "\n"
        puts "---------------------------------------------------------------------------"
        puts "Creating MRSessions for new subjects on XNAT"
        puts "---------------------------------------------------------------------------"
        set numMRIDs [llength $::XNAT(MRIDs)]
        for { set i 0 } { $i < $numMRIDs } { incr i } {
            set mrid [ lindex $::XNAT(MRIDs) $i ]
            set numSessions [llength $::XNATupload($mrid,MRSessionDirnameList)]
            for { set j 0} { $j < $numSessions } { incr j } {
                set sessionLabel [ lindex $::XNATupload($mrid,MRSessionDirnameList) $j ]

                #--- create experiment on xnat
                set expID [ AddExperiment $mrid $sessionLabel ]
                #--- if new experiment was added add metadata, sessionFiles & scan data
                if { $expID > 0 } {
                    AddExperimentMetadata $expID

                    if { $::XNATupload(AddAllScanData) } {
                        #--- if local directory exists, cd in and process.
                        if { [ file exists $sessionLabel] } {
                            if { [ file isdirectory $sessionLabel] } {
                                set cwd [ pwd ]
                                cd $sessionLabel
                                puts "\n"
                                puts "---------------------------------------------------------------------------"
                                puts "...Adding/updating sessionFiles for this MRSession"
                                puts "---------------------------------------------------------------------------"
                                AddSessionFiles $mrid $sessionLabel
                                puts "...done."
                                puts "\n"
                                puts "---------------------------------------------------------------------------"
                                puts "...Adding/updating scans and scan files for this MRSession"
                                puts "---------------------------------------------------------------------------"
                                AddAllScanData $mrid $sessionLabel
                                puts "...done."
                                # pop back to root directory
                                cd $cwd
                            }
                        }
                    } else {
                        puts "...>>>>>>ERROR: Can not find local directory $sessionLabel."
                    }
                } else {
                    puts "......... did not add experiment $sessionLabel for subject $mrid"
                }
            }
        }
    } else {
        puts "...(skipping creation of all MRSessions)"
    }

}

