/*
 * Decompiled with CFR 0.152.
 */
package org.hibernate.jdbc;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.sql.Connection;
import java.sql.SQLException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.ConnectionReleaseMode;
import org.hibernate.HibernateException;
import org.hibernate.Interceptor;
import org.hibernate.engine.SessionFactoryImplementor;
import org.hibernate.exception.JDBCExceptionHelper;
import org.hibernate.jdbc.Batcher;
import org.hibernate.jdbc.BorrowedConnectionProxy;
import org.hibernate.jdbc.JDBCContext;
import org.hibernate.util.JDBCExceptionReporter;

public class ConnectionManager
implements Serializable {
    private static final Log log = LogFactory.getLog(ConnectionManager.class);
    private transient SessionFactoryImplementor factory;
    private final Callback callback;
    private final ConnectionReleaseMode releaseMode;
    private transient Connection connection;
    private transient Connection borrowedConnection;
    private final boolean wasConnectionSupplied;
    private transient Batcher batcher;
    private transient Interceptor interceptor;
    private boolean isClosed;
    private transient boolean isFlushing;

    public ConnectionManager(SessionFactoryImplementor factory, Callback callback, ConnectionReleaseMode releaseMode, Connection connection, Interceptor interceptor) {
        this.factory = factory;
        this.callback = callback;
        this.interceptor = interceptor;
        this.batcher = factory.getSettings().getBatcherFactory().createBatcher(this, interceptor);
        this.connection = connection;
        this.wasConnectionSupplied = connection != null;
        this.releaseMode = this.wasConnectionSupplied ? ConnectionReleaseMode.ON_CLOSE : releaseMode;
    }

    private ConnectionManager(SessionFactoryImplementor factory, Callback callback, ConnectionReleaseMode releaseMode, Interceptor interceptor, boolean wasConnectionSupplied, boolean isClosed) {
        this.factory = factory;
        this.callback = callback;
        this.interceptor = interceptor;
        this.batcher = factory.getSettings().getBatcherFactory().createBatcher(this, interceptor);
        this.wasConnectionSupplied = wasConnectionSupplied;
        this.isClosed = isClosed;
        this.releaseMode = wasConnectionSupplied ? ConnectionReleaseMode.ON_CLOSE : releaseMode;
    }

    public SessionFactoryImplementor getFactory() {
        return this.factory;
    }

    public Batcher getBatcher() {
        return this.batcher;
    }

    public boolean isSuppliedConnection() {
        return this.wasConnectionSupplied;
    }

    public Connection getConnection() throws HibernateException {
        if (this.isClosed) {
            throw new HibernateException("connection manager has been closed");
        }
        if (this.connection == null) {
            this.openConnection();
        }
        return this.connection;
    }

    public boolean hasBorrowedConnection() {
        return this.borrowedConnection != null;
    }

    public Connection borrowConnection() {
        if (this.isClosed) {
            throw new HibernateException("connection manager has been closed");
        }
        if (this.isSuppliedConnection()) {
            return this.connection;
        }
        if (this.borrowedConnection == null) {
            this.borrowedConnection = BorrowedConnectionProxy.generateProxy(this);
        }
        return this.borrowedConnection;
    }

    public void releaseBorrowedConnection() {
        if (this.borrowedConnection != null) {
            this.borrowedConnection = null;
            BorrowedConnectionProxy.renderUnuseable(this.borrowedConnection);
        }
    }

    public boolean isAutoCommit() throws SQLException {
        return this.connection == null || this.connection.getAutoCommit();
    }

    public boolean isAggressiveRelease() {
        if (this.releaseMode == ConnectionReleaseMode.AFTER_STATEMENT) {
            return true;
        }
        if (this.releaseMode == ConnectionReleaseMode.AFTER_TRANSACTION) {
            boolean inAutoCommitState;
            try {
                inAutoCommitState = this.isAutoCommit() && !this.callback.isTransactionInProgress();
            }
            catch (SQLException e) {
                inAutoCommitState = true;
            }
            return inAutoCommitState;
        }
        return false;
    }

    private boolean isAggressiveReleaseNoTransactionCheck() {
        boolean inAutoCommitState;
        if (this.releaseMode == ConnectionReleaseMode.AFTER_STATEMENT) {
            return true;
        }
        try {
            inAutoCommitState = this.isAutoCommit();
        }
        catch (SQLException e) {
            inAutoCommitState = true;
        }
        return this.releaseMode == ConnectionReleaseMode.AFTER_TRANSACTION && inAutoCommitState;
    }

    public boolean isCurrentlyConnected() {
        return this.wasConnectionSupplied ? this.connection != null : !this.isClosed;
    }

    public void afterStatement() {
        if (this.isAggressiveRelease()) {
            if (this.isFlushing) {
                log.debug("skipping aggressive-release due to flush cycle");
            } else if (this.batcher.hasOpenResources()) {
                log.debug("skipping aggresive-release due to open resources on batcher");
            } else if (this.borrowedConnection != null) {
                log.debug("skipping aggresive-release due to borrowed connection");
            } else {
                this.aggressiveRelease();
            }
        }
    }

    public void afterTransaction() {
        if (this.isAfterTransactionRelease()) {
            this.aggressiveRelease();
        } else if (this.isAggressiveReleaseNoTransactionCheck() && this.batcher.hasOpenResources()) {
            log.info("forcing batcher resource cleanup on transaction completion; forgot to close ScrollableResults/Iterator?");
            this.batcher.closeStatements();
            this.aggressiveRelease();
        } else if (this.isOnCloseRelease()) {
            log.debug("transaction completed on session with on_close connection release mode; be sure to close the session to release JDBC resources!");
        }
        this.batcher.unsetTransactionTimeout();
    }

    private boolean isAfterTransactionRelease() {
        return this.releaseMode == ConnectionReleaseMode.AFTER_TRANSACTION;
    }

    private boolean isOnCloseRelease() {
        return this.releaseMode == ConnectionReleaseMode.ON_CLOSE;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Connection close() {
        try {
            Connection connection = this.cleanup();
            return connection;
        }
        finally {
            this.isClosed = true;
        }
    }

    public Connection manualDisconnect() {
        return this.cleanup();
    }

    public void manualReconnect() {
    }

    public void manualReconnect(Connection suppliedConnection) {
        this.connection = suppliedConnection;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Connection cleanup() throws HibernateException {
        this.releaseBorrowedConnection();
        if (this.connection == null) {
            log.trace("connection already null in cleanup : no action");
            return null;
        }
        try {
            log.trace("performing cleanup");
            this.batcher.closeStatements();
            Connection c = null;
            if (!this.wasConnectionSupplied) {
                this.closeConnection();
            } else {
                c = this.connection;
            }
            this.connection = null;
            Connection connection = c;
            return connection;
        }
        finally {
            this.callback.connectionCleanedUp();
        }
    }

    private void aggressiveRelease() {
        if (!this.wasConnectionSupplied) {
            log.debug("aggressively releasing JDBC connection");
            if (this.connection != null) {
                this.closeConnection();
            }
        }
    }

    private void openConnection() throws HibernateException {
        if (this.connection != null) {
            return;
        }
        log.debug("opening JDBC connection");
        try {
            this.connection = this.factory.getConnectionProvider().getConnection();
        }
        catch (SQLException sqle) {
            throw JDBCExceptionHelper.convert(this.factory.getSQLExceptionConverter(), sqle, "Cannot open connection");
        }
        this.callback.connectionOpened();
    }

    private void closeConnection() {
        if (log.isDebugEnabled()) {
            log.debug("releasing JDBC connection [" + this.batcher.openResourceStatsAsString() + "]");
        }
        try {
            if (!this.connection.isClosed()) {
                JDBCExceptionReporter.logAndClearWarnings(this.connection);
            }
            this.factory.getConnectionProvider().closeConnection(this.connection);
            this.connection = null;
        }
        catch (SQLException sqle) {
            throw JDBCExceptionHelper.convert(this.factory.getSQLExceptionConverter(), sqle, "Cannot release connection");
        }
    }

    public void flushBeginning() {
        log.trace("registering flush begin");
        this.isFlushing = true;
    }

    public void flushEnding() {
        log.trace("registering flush end");
        this.isFlushing = false;
        this.afterStatement();
    }

    public boolean isReadyForSerialization() {
        return this.wasConnectionSupplied ? this.connection == null : !this.batcher.hasOpenResources();
    }

    private void writeObject(ObjectOutputStream oos) throws IOException {
        if (!this.isReadyForSerialization()) {
            throw new IllegalStateException("Cannot serialize a ConnectionManager while connected");
        }
        oos.writeObject(this.factory);
        oos.writeObject(this.interceptor);
        oos.defaultWriteObject();
    }

    private void readObject(ObjectInputStream ois) throws IOException, ClassNotFoundException {
        this.factory = (SessionFactoryImplementor)ois.readObject();
        this.interceptor = (Interceptor)ois.readObject();
        ois.defaultReadObject();
        this.batcher = this.factory.getSettings().getBatcherFactory().createBatcher(this, this.interceptor);
    }

    public void serialize(ObjectOutputStream oos) throws IOException {
        oos.writeBoolean(this.wasConnectionSupplied);
        oos.writeBoolean(this.isClosed);
    }

    public static ConnectionManager deserialize(ObjectInputStream ois, SessionFactoryImplementor factory, Interceptor interceptor, ConnectionReleaseMode connectionReleaseMode, JDBCContext jdbcContext) throws IOException {
        return new ConnectionManager(factory, jdbcContext, connectionReleaseMode, interceptor, ois.readBoolean(), ois.readBoolean());
    }

    public static interface Callback {
        public void connectionOpened();

        public void connectionCleanedUp();

        public boolean isTransactionInProgress();
    }
}

