
XML = """<?xml version="1.0" encoding="utf-8"?>
<executable>

  <category>Python Modules</category>
  <title>Python Reslice As Volume</title>
  <description>
Reslice a volume based on the geometry of another volume.
</description>
  <version>1.0</version>
  <documentation-url></documentation-url>
  <license></license>
  <contributor>Luca Antiga and Daniel Blezek</contributor>

  <parameters>
    <label>Reslice Parameters</label>
    <description>Parameters for the reslice filter</description>

    <string-enumeration>
      <name>interpolation</name>
      <longflag>interpolation</longflag>
      <description>Interpolation mode</description>
      <label>Interpolation Mode</label>
      <default>linear</default>
      <element>nearest neighbor</element>
      <element>linear</element>
      <element>cubic</element>
    </string-enumeration>

    <double>
      <name>background</name>
      <longflag>background</longflag>
      <description>Background level for "outside" values</description>
      <label>Background</label>
      <default>0.0</default>
    </double>

  </parameters>

  <parameters>
    <label>IO</label>
    <description>Input/output parameters</description>
    <image>
      <name>inputVolume</name>
      <label>Input Volume</label>
      <channel>input</channel>
      <index>0</index>
      <description>Input volume to be resampled</description>
    </image>
    <image>
      <name>referenceVolume</name>
      <label>Reference Volume</label>
      <channel>input</channel>
      <index>1</index>
      <description>Reference volume defining the geometry</description>
    </image>
    <image>
      <name>outputVolume</name>
      <label>Output Volume</label>
      <channel>output</channel>
      <index>2</index>
      <description>Output resampled volume</description>
    </image>
  </parameters>

</executable>
"""


def Execute (inputVolume, referenceVolume, outputVolume, background=0.0, interpolation="linear"):

    Slicer = __import__("Slicer")
    slicer = Slicer.Slicer()
    scene = slicer.MRMLScene

    inputVolume = scene.GetNodeByID(inputVolume)
    referenceVolume = scene.GetNodeByID(referenceVolume)
    outputVolume = scene.GetNodeByID(outputVolume)

    resliceMatrix = slicer.vtkMatrix4x4.New()
    inputVolume.GetIJKToRASMatrix(resliceMatrix)

    for i in range(3):
        v = [resliceMatrix.GetElement(0,i), 
             resliceMatrix.GetElement(1,i), 
             resliceMatrix.GetElement(2,i)]
        # slicer.vtkMath.Normalize(v) doesn't work
        # vnorm = (v[0]**2 + v[1]**2 + v[2]**2)**0.5
        resliceMatrix.SetElement(0,i,v[0]/vnorm)
        resliceMatrix.SetElement(1,i,v[1]/vnorm)
        resliceMatrix.SetElement(2,i,v[2]/vnorm)
    resliceMatrix.SetElement(0,3,0.0)
    resliceMatrix.SetElement(1,3,0.0)
    resliceMatrix.SetElement(2,3,0.0)

    resliceMatrix.Invert()

    referenceMatrix = slicer.vtkMatrix4x4.New()
    referenceVolume.GetIJKToRASMatrix(referenceMatrix)

    resliceMatrix.Multiply4x4(resliceMatrix,referenceMatrix,resliceMatrix)
#    resliceMatrix.Invert()

    resliceTransform = slicer.vtkTransform.New()
    resliceTransform.SetMatrix(resliceMatrix)

    reslice = slicer.vtkImageReslice.New()
    reslice.SetInput(inputVolume.GetImageData())
    reslice.SetInformationInput(referenceVolume.GetImageData())
    reslice.SetBackgroundLevel(background)
    reslice.SetResliceTransform(resliceTransform)
    if interpolation == "nearest neighbor":
        reslice.SetInterpolationModeToNearestNeighbor()
    elif interpolation == "linear":
        reslice.SetInterpolationModeToLinear()
    elif interpolation == "cubic":
        reslice.SetInterpolationModeToCubic()
    reslice.Update()

    outputVolume.SetAndObserveImageData(reslice.GetOutput())
 
    outputMatrix = slicer.vtkMatrix4x4.New()
    referenceVolume.GetIJKToRASMatrix(outputMatrix)
    outputVolume.SetIJKToRASMatrix(outputMatrix)

    return

